<?php
/**
 * Miscellaneous template tags and template utility functions
 *
 * These functions are for use throughout the theme's various template files.
 * This file is loaded via the 'after_setup_theme' hook at priority '10'
 *
 * @package    Magazine News Byte
 * @subpackage Theme Premium
 */

/**
 * Apply Sidebar Layout for Archives and Blog page
 * Add it at earliest priority as this is core option
 *
 * @since 1.0
 * @param bool $sidebar
 * @return string
 */

if ( !function_exists( 'magnb_layout_extension' ) ):
function magnb_layout_extension( $sidebar ) {
	if ( is_singular() && !is_front_page() ) { // Page got 'sidebar_type' set before assigning it as front page : This option is not visible on Edit Page after it is set as frontpage
		$type = hoot_get_metaoption( 'sidebar_type' );
		if ( 'custom' === $type )
			$sidebar = hoot_get_metaoption( 'sidebar' );
	}
	return $sidebar;
}
endif;
add_filter( 'magnb_layout', 'magnb_layout_extension', 5 );

/**
 * Display code for megamenu icon
 *
 * @since 1.0
 * @param string   $title The menu item's title.
 * @param WP_Post  $item  The current menu item.
 * @param stdClass $args  An object of wp_nav_menu() arguments.
 * @param int      $depth Depth of menu item. Used for padding.
 * @return string
 */
if ( !function_exists( 'magnb_megamenu_displayicon' ) ):
function magnb_megamenu_displayicon( $title, $item, $args, $depth ) {
	$hoot_megamenu_item = ( isset( $item->ID ) ) ? get_post_meta( $item->ID, '_menu-item-hoot_megamenu', true ) : array();
	if ( isset( $hoot_megamenu_item[ 'hoot_icon' ] ) && !empty( $hoot_megamenu_item[ 'hoot_icon' ] ) )
		$title = '<i class="hoot-megamenu-icon ' . hoot_sanitize_fa( $hoot_megamenu_item[ 'hoot_icon' ] ) . '"></i> <span>' . $title . '</span>';
	return $title;
}
endif;
add_filter( 'nav_menu_item_title', 'magnb_megamenu_displayicon', 3, 4 ); // Hook into filter early on to display menuicon within the <span class="menu-title"> added by core theme at priority 5

/**
 * Hide or display meta information for current page/post
 *
 * @since 1.0
 * @param string $hide
 * @param string $context
 * @return string
 */
if ( !function_exists( 'magnb_hide_meta_info' ) ):
function magnb_hide_meta_info( $hide, $context ) {
	$override = hoot_get_metaoption( 'meta_hide_info' );
	return ( $override ) ? $override : $hide;
}
endif;
add_filter( 'magnb_hide_meta', 'magnb_hide_meta_info', 5, 2 );

/**
 * Hide or display Page Loop Area for current page/post
 *
 * @since 1.0
 * @param string $value
 * @return string
 */
if ( !function_exists( 'magnb_loop_meta_display_title' ) ):
function magnb_loop_meta_display_title( $value ) {
	if ( is_page() || is_singular( 'post' ) )
		$id = get_queried_object_id();
	elseif ( is_home() && !is_front_page() )
		$id = get_option( 'page_for_posts' );
	// elseif ( current_theme_supports( 'woocommerce' ) && is_shop() )
	// 	$id = get_option( 'woocommerce_shop_page_id' );
	else
		return $value;
	return hoot_get_metaoption( 'display_loop_meta', $id );
}
endif;
add_filter( 'magnb_loop_meta_display_title', 'magnb_loop_meta_display_title', 5 );

/**
 * Hide or display Page Loop Area for Woocommerce Products (archives) Page (shop/taxonomies)
 *
 * @since 1.0
 * @param string $value
 * @return string
 */
if ( !function_exists( 'magnb_wooloop_meta_display_title' ) ):
function magnb_wooloop_meta_display_title( $value ) {
	if ( current_theme_supports( 'woocommerce' ) && is_shop() ) {
		$id = get_option( 'woocommerce_shop_page_id' );
		return hoot_get_metaoption( 'display_loop_meta', $id );
	}
	return $value;
}
endif;
add_filter( 'magnb_woo_loop_meta_display_title', 'magnb_wooloop_meta_display_title', 5 );

/**
 * Page Loop Area
 *
 * @since 1.0
 * @param string $value
 * @param string $location
 * @param string $context
 * @return string
 */
if ( !function_exists( 'magnb_loop_meta_pre_title_content' ) ):
function magnb_loop_meta_pre_title_content( $value, $location, $context ) {
	if ( is_page() || is_singular( 'post' ) )
		$id = get_queried_object_id();
	elseif ( is_home() && !is_front_page() )
		$id = get_option( 'page_for_posts' );
	elseif ( current_theme_supports( 'woocommerce' ) && is_shop() )
		$id = get_option( 'woocommerce_shop_page_id' );
	else
		return $value;
	return hoot_get_metaoption( 'pre_title_content', $id );
}
endif;
add_filter( 'magnb_pre_title_content', 'magnb_loop_meta_pre_title_content', 5, 3 );

/**
 * Page Loop Area
 *
 * @since 1.0
 * @param string $value
 * @param string $location
 * @param string $context
 * @return string
 */
if ( !function_exists( 'magnb_loop_meta_pre_title_content_stretch' ) ):
function magnb_loop_meta_pre_title_content_stretch( $value, $location, $context ) {
	if ( is_page() || is_singular( 'post' ) )
		$id = get_queried_object_id();
	elseif ( is_home() && !is_front_page() )
		$id = get_option( 'page_for_posts' );
	elseif ( current_theme_supports( 'woocommerce' ) && is_shop() )
		$id = get_option( 'woocommerce_shop_page_id' );
	else
		return $value;
	return hoot_get_metaoption( 'pre_title_content_stretch', $id );
}
endif;
add_filter( 'magnb_pre_title_content_stretch', 'magnb_loop_meta_pre_title_content_stretch', 5, 3 );

/**
 * Page Loop Area
 *
 * @since 1.0
 * @param string $value
 * @param string $location
 * @param string $context
 * @return string
 */
if ( !function_exists( 'magnb_loop_meta_pre_title_content_post' ) ):
function magnb_loop_meta_pre_title_content_post( $value, $location, $context ) {
	if ( is_page() || is_singular( 'post' ) )
		$id = get_queried_object_id();
	elseif ( is_home() && !is_front_page() )
		$id = get_option( 'page_for_posts' );
	elseif ( current_theme_supports( 'woocommerce' ) && is_shop() )
		$id = get_option( 'woocommerce_shop_page_id' );
	else
		return $value;
	return hoot_get_metaoption( 'pre_title_content_post', $id );
}
endif;
add_filter( 'magnb_pre_title_content_post', 'magnb_loop_meta_pre_title_content_post', 5, 3 );

/**
 * Allow users to determine WooCommerce Page Layouts
 *
 * @since 1.0
 * @param bool
 * @return bool
 */
if ( !function_exists( 'magnb_woo_pages_force_nosidebar' ) ):
function magnb_woo_pages_force_nosidebar( $value ) {
	return false;
}
endif;
add_filter( 'magnb_woo_pages_force_nosidebar', 'magnb_woo_pages_force_nosidebar', 5 );

/**
 * Display custom 404 page content
 *
 * @since 1.0
 * @return void
 */
if ( !function_exists( 'magnb_custom404' ) ):
function magnb_custom404() {
	if ( is_404() && 'custom' == hoot_get_mod('404_page') ) {
		$page_404 = intval( hoot_get_mod('404_custom_page') );
		if ( !empty( $page_404 ) ) {
			remove_action( 'magnb_404_content', 'magnb_display_404_content', 5 );
			add_action( 'magnb_404_content', 'magnb_display_custom404_content', 5 );
			add_filter( 'magnb_display_404_title', 'magnb_display_custom404_content_title' );
		}
	}
}
endif;
add_action( 'wp', 'magnb_custom404' );

if ( !function_exists( 'magnb_display_custom404_content' ) ):
function magnb_display_custom404_content() {
	$page_404 = intval( hoot_get_mod('404_custom_page') );
	$post = get_post( $page_404 );
	if ( isset( $post->post_content ) )
		echo apply_filters( 'the_content', $post->post_content );
}
endif;

if ( !function_exists( 'magnb_display_custom404_content_title' ) ):
function magnb_display_custom404_content_title(){
	return false;
}
endif;

/**
 * Loop pagination function. This can be used for custom loops to create pagination
 * Uses WordPress's core 'paginate_links' function
 * @NU
 *
 * @since 1.0
 * @access public
 * @param array $args Arguments to customize how the page links are output.
 * @return string $page_links
 */
if ( !function_exists( 'magnb_loop_pagination' ) ):
function magnb_loop_pagination( $args = array(), $custom_query = array(), $base_url = '' ) {
	global $wp_rewrite, $wp_query;

	/* Use query */
	$query = ( ! empty( $custom_query ) ) ? $custom_query : $wp_query;

	/* Set base url */
	$base_url = ( !empty( $base_url ) ) ? esc_url( $base_url ) : get_pagenum_link();

	/* If there's not more than one page, return nothing. */
	if ( 1 >= $query->max_num_pages )
		return;

	/* Get the current page. */
	$current = ( get_query_var( 'paged' ) ? absint( get_query_var( 'paged' ) ) : 1 );

	/* Get the max number of pages. */
	$max_num_pages = intval( $query->max_num_pages );

	/* Get the pagination base. */
	$pagination_base = $wp_rewrite->pagination_base;

	/* Set up some default arguments for the paginate_links() function. */
	$defaults = array(
		'base'         => add_query_arg( 'paged', '%#%', $base_url ),
		'format'       => '',
		'total'        => $max_num_pages,
		'current'      => $current,
		'prev_next'    => true,
		'prev_text'  => __( '&laquo; Previous', 'magazine-news-byte-premium' ), // This is the WordPress default.
		'next_text'  => __( 'Next &raquo;', 'magazine-news-byte-premium' ), // This is the WordPress default.
		'show_all'     => false,
		'end_size'     => 1,
		'mid_size'     => 1,
		'add_fragment' => '',
		'type'         => 'plain',

		// Begin magnb_loop_pagination() arguments.
		'before'       => '<nav class="pagination loop-pagination">',
		'after'        => '</nav>',
		'echo'         => true,
	);

	/* Add the $base argument to the array if the user is using permalinks. */
	if ( $wp_rewrite->using_permalinks() && !is_search() )
		$defaults['base'] = user_trailingslashit( trailingslashit( $base_url ) . "{$pagination_base}/%#%" );

	/* Allow developers to overwrite the arguments with a filter. */
	$args = apply_filters( 'magnb_loop_pagination_args', $args );

	/* Merge the arguments input with the defaults. */
	$args = wp_parse_args( $args, $defaults );

	/* Don't allow the user to set this to an array. */
	if ( 'array' == $args['type'] )
		$args['type'] = 'plain';

	/* Get the paginated links. */
	$page_links = paginate_links( $args );

	/* Remove 'page/1' from the entire output since it's not needed. */
	$page_links = preg_replace( 
		array( 
			"#(href=['\"].*?){$pagination_base}/1(['\"])#",  // 'page/1'
			"#(href=['\"].*?){$pagination_base}/1/(['\"])#", // 'page/1/'
			"#(href=['\"].*?)\?paged=1(['\"])#",             // '?paged=1'
			"#(href=['\"].*?)&\#038;paged=1(['\"])#"         // '&#038;paged=1'
		), 
		'$1$2', 
		$page_links 
	);

	/* Wrap the paginated links with the $before and $after elements. */
	$page_links = $args['before'] . $page_links . $args['after'];

	/* Allow devs to completely overwrite the output. */
	$page_links = wp_kses_post( apply_filters( 'magnb_loop_pagination', $page_links ) );

	/* Return the paginated links for use in themes. */
	if ( $args['echo'] )
		echo $page_links;
	else
		return $page_links;
}
endif;

/**
 * Insert Custom Javascript code by the user in either header or footer
 *
 * @since 1.0
 * @return void
 */
if ( !function_exists( 'magnb_premium_custom_user_js' ) ):
function magnb_premium_custom_user_js() {
	$custom_js = trim( hoot_get_mod( 'custom_js', '' ) );
	if ( !empty( $custom_js ) ) {
		if ( !strpos( $custom_js, '<script>' ) && !strpos( $custom_js, '</script>' ) ) {
			$start = '<script>'; $end = '</script>';
		} else $start = $end = '';
		// $custom_js = preg_replace( '/^' . preg_quote('<script>', '/') . '/', '', $custom_js);
		// $custom_js = preg_replace( '/' . preg_quote('</script>', '/') . '$/', '', $custom_js);
		echo "\n" . $start . htmlspecialchars_decode( $custom_js ) . $end . "\n";
	}
}
endif;
if ( hoot_get_mod( 'custom_js_inheader' ) )
	add_action( 'wp_head', 'magnb_premium_custom_user_js' );
else
	add_action( 'wp_footer', 'magnb_premium_custom_user_js' );