<?php
/**
 * Meta Options Extension.
 * This file is loaded at 'after_setup_theme' hook with 4 priority.
 *
 * @package    Magazine News Byte
 * @subpackage Library Premium
 */

/** Include Meta Options files **/
require_once( hoot_data()->premium_libdir . 'metaoptions/functions.php' );
require_once( hoot_data()->premium_libdir . 'metaoptions/interface.php' );

/**
 * Class wrapper with useful methods for interacting with the hoot customizer.
 * 
 * @since 3.0.0
 */
final class Hoot_Metaoptions {

	/**
	 * Holds the instance of this class.
	 *
	 * @since 3.0.0
	 * @access private
	 * @var object
	 */
	private static $instance;

	/**
	 * Metabox Options Array
	 *
	 * @since 3.0.0
	 * @access private
	 * @var array
	 */
	private $meta_options = array();

	/**
	 * Protected constructor to prevent creating a new instance of the
	 * Singleton from outside of this class.
	 *
	 * @since 3.0.0
	 */
	protected function __construct() {

		if ( !is_admin() )
			return;

		/* Hook in to create new metabozes */
		add_action( 'init', array( $this, 'add_meta_boxes' ) );

	}

	/*
	 * Hook in to create new metaboxes
	 *
	 * @since 3.0.0
	 */
	function add_meta_boxes() {
		if ( !empty( $this->meta_options ) ) {

			/* Hook to action 'add_meta_boxes' for required post types */
			foreach ( $this->meta_options as $post_type => $options )
				add_action( "add_meta_boxes_{$post_type}", array( $this, 'add_meta_box' ), 10, 1 );

			/* Save meta values properly */
			add_action( 'save_post', array( $this, 'save_postdata' ) );

			/* Add the required scripts and styles */
			add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_styles_scripts' ) );

		}
	}

	/**
	 * Create Meta Boxes.
	 *
	 * @since 3.0.0
	 * @param object $post
	 */
	function add_meta_box( $post ) {

		// Get post type. Return false or string.
		$post_type = get_post_type( $post );

		// Add all metaboxes for $post_type
		if ( $post_type &&
			 isset( $this->meta_options[ $post_type ] ) &&
			 is_array( $this->meta_options[ $post_type ] ) &&
			 current_user_can( 'edit_post_meta', $post->ID )
			 ) :
			foreach ( $this->meta_options[ $post_type ] as $metabox_id => $metabox_settings ) {

				$metabox_settings = wp_parse_args( $metabox_settings, array(
										'title'    => __( 'Options', 'magazine-news-byte-premium' ),
										'context'  => 'normal',  // 'normal', 'advanced', or 'side'
										'priority' => 'high',    // 'high', 'core', 'default' or 'low'
										'options'  => array(),
									) );

				$metabox_id = sanitize_html_class( $metabox_id );
				add_meta_box( 'hoot-metabox-' . $metabox_id,
							  esc_html( $metabox_settings['title'] ),
							  array( $this, 'meta_box' ),        // Callback Function
							  $post_type,
							  $metabox_settings['context'],
							  $metabox_settings['priority'],
							  array( $post_type, $metabox_id )   // Callback Args - second parameter passed to callback fn
							  );

			}
		endif;

	}

	/**
	 * Builds out the meta box.
	 *
	 * @since 3.0.0
	 * @param object $post
	 * @param array $args arguments passed to callback function
	 */
	function meta_box( $post, $args ) {
		$post_type = $args['args'][0];
		$metabox_id = $args['args'][1];
		$meta_values = get_post_meta( $post->ID, '_hoot_meta_' . $metabox_id , true );
		$meta_values = apply_filters( 'hoot_admin_metabox_value', $meta_values, $post->ID, $post_type, $metabox_id );

		wp_nonce_field( basename( __FILE__ ), 'hoot-metaoptions-nonce' ); ?>

		<label class="screen-reader-text" for="excerpt"><?php echo $this->meta_options[ $post_type ][ $metabox_id ]['title'] ?></label>

		<div id="<?php echo ( sanitize_html_class( 'hoot-metaoptions-' . $metabox_id ) ); ?>" class="hoot-metaoptions">
			<?php
			if ( !empty ( $this->meta_options[ $post_type ][ $metabox_id ]['options'] ) ) {
				$hoot_metaoptions_interface = Hoot_Metaoptions_Interface::get_instance();
				$hoot_metaoptions_interface->build(
					$this->meta_options[ $post_type ][ $metabox_id ]['options'],
					$meta_values,
					'hoot' . '-' . $metabox_id
					);
			}
			?>
		</div>

	<?php
	}

	/**
	 * Save the meta box.
	 *
	 * @since 3.0.0
	 * @access public
	 * @param int $post_id The ID of the current post being saved.
	 * @param object $post The post object currently being saved.
	 */
	function save_postdata( $post_id, $post = '' ){

		/* Verify nonce */
		if ( ! isset( $_POST['hoot-metaoptions-nonce'] ) || ! wp_verify_nonce( $_POST['hoot-metaoptions-nonce'], basename( __FILE__ ) ) )
			return $post_id;

		/* verify if this is an auto save routine. If it is our form has not been submitted, so we dont want to do anything */
		if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE )
			return $post_id;

		/* Security - Check permissions */
		if ( isset( $_POST['post_type'] ) && 'page' == $_POST['post_type'] ) {
			if ( ! current_user_can( 'edit_page', $post_id ) )
				return $post_id;
		} else {
			if ( ! current_user_can( 'edit_post', $post_id ) )
				return $post_id;
		}

		/* Check User Permissions */
		if ( !current_user_can( 'edit_post_meta', $post_id ) )
			return $post_id;

		// OK, we're authenticated

		/* Sanitize and Save (array) */
		if ( isset( $_POST['post_type'] ) && isset( $this->meta_options[ $_POST['post_type'] ] ) ):
			foreach ( $this->meta_options[ $_POST['post_type'] ] as $metabox_id => $metabox_settings ) {

				$metabox_id = sanitize_html_class( $metabox_id );
				$values = isset( $_POST[ 'hoot' . '-' . $metabox_id ] ) ? $_POST[ 'hoot' . '-' . $metabox_id ] : array();
				$fields = !empty( $metabox_settings['options'] ) ? $metabox_settings['options'] : array();
				if ( $values && $fields ) {
					$clean = hoot_sanitize_get_meta( $values, $fields );
					if ( !empty( $clean ) )
						update_post_meta( $post_id, '_hoot_meta_' . $metabox_id, $clean );
				}

			}
		endif;

	}

	/**
	 * Loads the required stylesheets and scripts
	 *
	 * @since 3.0.0
	 */
	function enqueue_admin_styles_scripts( $hook ) {
		if ( $hook == 'post-new.php' || $hook == 'post.php' ) :
			$screen = get_current_screen();
			$post_type = $screen->post_type;
			$supported_post_types = array_keys( $this->meta_options );
			if ( in_array( $post_type, $supported_post_types ) ) :

				// Enqueue Styles
				wp_enqueue_style( 'font-awesome' );
				wp_enqueue_style( 'wp-color-picker' );
				$style_uri = hoot_locate_style( hoot_data()->premium_liburi . 'css/metaoptions' );
				wp_enqueue_style( 'hoot-metaoptions', $style_uri, array(),  hoot_data()->hoot_version );

				// Enqueue Scripts
				$script_uri = hoot_locate_script( hoot_data()->premium_liburi . 'js/metaoptions' );
				wp_enqueue_script( 'hoot-metaoptions', $script_uri, array( 'jquery', 'wp-color-picker' ), hoot_data()->hoot_version, true );

				/* Load Uploader javascript */
				$load = apply_filters( 'hoot_metaoptions_upload', false );
				if ( $load ) {
					if ( function_exists( 'wp_enqueue_media' ) )
						wp_enqueue_media();
					$script_uri = hoot_locate_script( hoot_data()->premium_liburi . 'js/media-uploader' );
					wp_enqueue_script( 'hoot-metaoptions-media', $script_uri, array( 'jquery' ), hoot_data()->hoot_version );
					wp_localize_script( 'hoot-metaoptions-media', '_hoot_metaoptions_data', array(
						'upload' => __( 'Upload', 'magazine-news-byte-premium' ),
						'remove' => __( 'Remove', 'magazine-news-byte-premium' )
					) );
				}

			endif;
			do_action( 'hoot_metaoptions_enqueue', $hook, $post_type, $supported_post_types );
		endif;
	}

	/**
	 * Add meta option to meta_options Array
	 *
	 * @since 3.0.0
	 * @access public
	 * @return void
	 */
	public function add_options( $meta_options = array() ) {
		$meta_options = apply_filters( 'hoot_metaoptions_array' , $meta_options );
		$this->meta_options = array_merge( $this->meta_options, $meta_options );
	}

	/**
	 * Get meta_options Array
	 *
	 * @since 3.0.0
	 * @access public
	 * @return array
	 */
	public function get_options( $post_type = '', $meta_key = '' ) {
		if ( !empty( $post_type ) ) {
			if ( isset( $this->meta_options[ $post_type ] ) ) {
				if ( !empty( $meta_key ) )
					return ( isset( $this->meta_options[ $post_type ][ $meta_key ] ) ) ? $this->meta_options[ $post_type ][ $meta_key ] : array();
				else
					return $this->meta_options[ $post_type ];
			} else return array();
		}
		return $this->meta_options;
	}

	/**
	 * Returns the instance.
	 *
	 * @since 3.0.0
	 * @access public
	 * @return object
	 */
	public static function get_instance() {
		if ( is_null( self::$instance ) ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

}

/** Initialize and run **/
Hoot_Metaoptions::get_instance();