<?php
/**
 * Theme Manager Extension
 * This file is loaded at 'after_setup_theme' hook with 4 priority.
 *
 * @package    Magazine News Byte
 * @subpackage Library Premium
 */

/**
 * Theme Manager class. This wraps everything up nicely.
 *
 * @since 3.0.0
 */
class Hoot_Manager {

	/**
	 * Holds the instance of this class.
	 *
	 * @since 3.0.0
	 * @access private
	 * @var object
	 */
	private static $instance;

	/**
	 * Supported modules
	 *
	 * @since 3.0.0
	 * @access public
	 * @var array
	 */
	public $support = array();

	/**
	 * Initialize everything
	 * 
	 * @since 3.0.0
	 * @access public
	 * @return void
	 */
	public function __construct() {

		/* Get Modules Support */
		$this->support = apply_filters( 'hoot_manager_modules', array( 'license', 'import', 'export' ) );

		/* Add the Theme Manager page */
		if ( is_admin() && current_user_can( 'edit_theme_options' ) ) {
			add_action( 'admin_menu', array( $this, 'manager_add_page' ) );
			add_action( 'admin_menu', array( $this, 'reorder_custom_options_page' ), 9995 );
		}

		/* Add the required scripts and styles */
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_wp_styles_scripts' ) );

		/* Add modules */
		add_action( 'init', array( $this, 'add_modules' ) );

		/* Remove autoupdates if license not active */
		$active = get_option( hoot_data()->basetemplate_slug . '_active_status', false );
		if ( 'valid' !== $active )
			add_filter( 'theme_auto_update_setting_template', array( $this, 'auto_update_setting_template' ) );

	}

	/**
	 * Loads the required stylesheets and scripts
	 *
	 * @since 3.0.0
	 */
	function enqueue_wp_styles_scripts( $hook ) {
		if ( 'appearance_page_hoot-manager' == $hook ) {
			$style_uri = hoot_locate_style( hoot_data()->premium_liburi . 'css/theme-manager' );
			wp_enqueue_style( 'hoot-manager', $style_uri, array(),  hoot_data()->hoot_version );
		}
	}

	/**
	 * Add Theme Manager Page
	 *
	 * @since 3.0.0
	 */
	function manager_add_page() {

		$name = apply_filters( 'hoot_manager_name', __( 'Hoot Theme Manager', 'magazine-news-byte-premium' ) );
		add_theme_page( $name, $name, 'manage_options', 'hoot-manager', array( $this, 'manager_do_page' ) );

	}

	/**
	 * Reorder subpage called "Theme Options" in the appearance menu.
	 *
	 * @since 3.0.0
	 */
	function reorder_custom_options_page() {
		global $submenu;
		$menu_slug = 'hoot-manager';
		$index = '';

		if ( !isset( $submenu['themes.php'] ) ) {
			// probably current user doesn't have this item in menu
			return;
		}

		foreach ( $submenu['themes.php'] as $key => $sm ) {
			if ( $sm[2] == $menu_slug ) {
				$index = $key;
				break;
			}
		}

		if ( ! empty( $index ) ) {
			//$item = $submenu['themes.php'][ $index ];
			//unset( $submenu['themes.php'][ $index ] );
			//array_splice( $submenu['themes.php'], 1, 0, array($item) );
			/* array_splice does not preserve numeric keys, so instead we do our own rearranging. */
			$smthemes = array();
			foreach ( $submenu['themes.php'] as $key => $sm ) {
				if ( $key != $index ) {
					$setkey = $key;
					// Find next available position if current one is taken
					for ( $i = $key; $i < 1000; $i++ ) {
						if( !isset( $smthemes[$i] ) ) {
							$setkey = $i;
							break;
						}
					}
					$smthemes[ $setkey ] = $sm;
					if ( $sm[1] == 'customize' ) { // if ( $sm[2] == 'themes.php' ) {
						$smthemes[ $setkey + 1 ] = $submenu['themes.php'][ $index ];
					}
				}
			}
			$submenu['themes.php'] = $smthemes;
		}

	}

	/**
	 * Print Theme Manager
	 *
	 * @since 3.0.0
	 */
	function manager_do_page() {
		if ( !current_user_can( 'manage_options' ) )  {
			wp_die( __( 'You do not have sufficient permissions to access this page.', 'magazine-news-byte-premium' ) );
		}

		?>
		<div id="hoot-manager" class="wrap">
			<h1><?php echo apply_filters( 'hoot_manager_name', __( 'Theme Manager', 'magazine-news-byte-premium' ) ); ?></h1>
			<?php do_action( 'hoot_manager_page' ); ?>
		</div>
		<?php

	}

	/**
	 * Add Theme Manager Modules
	 *
	 * @since 3.0.0
	 */
	function add_modules() {

		/** Auto Updater **/
		if ( in_array( 'license', $this->support ) ) {
			require_once( hoot_data()->premium_libdir . 'manager/manager-license.php' );
			new Hoot_Manager_License();
		}

		if ( is_admin() && current_user_can( 'edit_theme_options' ) ) {
			/** Import **/
			if ( in_array( 'import', $this->support ) ) {
				require_once( hoot_data()->premium_libdir . 'manager/manager-import.php' );
				new Hoot_Manager_Import();
			}
			/** Export **/
			if ( in_array( 'export', $this->support ) ) {
				require_once( hoot_data()->premium_libdir . 'manager/manager-export.php' );
				new Hoot_Manager_Export();
			}
		}

	}

	/**
	 * Remove autoupdates if license not active
	 *
	 * @since 3.0.0
	 */
	function auto_update_setting_template( $template ) {
		$template_slug = strtolower( preg_replace( '/[^a-zA-Z0-9]+/', '-', trim( hoot_data()->template_name ) ) );
		$theme_slug = ( is_child_theme() ) ? strtolower( preg_replace( '/[^a-zA-Z0-9]+/', '-', trim( hoot_data()->childtheme_name ) ) ) : $template_slug;
		$text = __( 'Please activate license key to enable Auto-updates for this theme.', 'magazine-news-byte-premium' );
		// @TS $template_slug $theme_slug {{ data.id }}
		return "<# if ( [ '" . $template_slug . "', '" . $theme_slug . "' ].includes( data.id ) ) { #>
			<p>$text</p>
			<# } else { #>
			$template
			<# } #>";
	}

	/**
	 * Returns the instance.
	 *
	 * @since 3.0.0
	 * @access public
	 * @return object
	 */
	public static function get_instance() {

		if ( !self::$instance )
			self::$instance = new self;

		return self::$instance;
	}

}

/* Initialize class */
global $hoot_manager;
$hoot_manager = Hoot_Manager::get_instance();